-- currenttime()    = TimeStamp Without Time Zone
-- current_timestap = TimeStamp With Time Zone
-- Info: "WITHOUT TIME ZONE" nicht mehr nötig - TIMESTAMP wurde mit Postgre 7.3 dem SQL-Standard angepasst und ist jetzt standardmäßig ohne TIMEZONE
CREATE OR REPLACE FUNCTION currenttime()
RETURNS timestamp AS $$
  SELECT date_trunc(
      'second',
      statement_timestamp()::timestamp
  )
$$ LANGUAGE sql;


-- wegen Index-Nutzung: https://stackoverflow.com/questions/68497605/postgresql-not-using-index-for-queries-using-current-user
CREATE OR REPLACE FUNCTION TSystem.currentuser()
RETURNS varchar AS $$
   SELECT current_user::varchar
$$ LANGUAGE sql STABLE;

CREATE OR REPLACE FUNCTION TSystem.current_user()
RETURNS varchar AS $$
   SELECT current_user::varchar
$$ LANGUAGE sql STABLE;


-- Wie DISTINCT nur unterschiedliche Zeilen
CREATE OR REPLACE FUNCTION tsystem.array_unique( _array anyarray )
RETURNS anyarray AS $$
    SELECT array( SELECT DISTINCT unnest( _array ) );
$$ LANGUAGE sql IMMUTABLE;

-- Besonderheiten von Arrays berücksichtigen {} und {null} siehe Beispiele unter der Funktion
CREATE OR REPLACE FUNCTION tsystem.array_null( _array anyarray )
RETURNS boolean AS $$
    SELECT NOT EXISTS( SELECT * FROM unnest( _array ) WHERE unnest IS NOT null);
$$ LANGUAGE sql IMMUTABLE;
-- SELECT tsystem.array_null('{}'::int[]) => true
-- SELECT tsystem.array_null('{null}'::int[]) => true
-- SELECT tsystem.array_null('{1}'::int[]) => false


-- alternative zur standard array syntax array[1,2,3]::int[]
-- da delphi/PgDac keine arrays an queries binden kann
CREATE OR REPLACE FUNCTION tsystem.array__create(
     VARIADIC _elements anyarray
  )
  RETURNS anyarray AS $$
      SELECT _elements
  $$ LANGUAGE sql IMMUTABLE;

CREATE OR REPLACE FUNCTION tsystem.array__create__from__variadic(
  VARIADIC _elements varchar[]
  )
  RETURNS varchar[]
  AS $$
     SELECT _elements;
  $$ LANGUAGE sql IMMUTABLE;

CREATE OR REPLACE FUNCTION tsystem.array__create__from__variadic(
  VARIADIC _elements integer[]
  )
  RETURNS integer[]
  AS $$
     SELECT _elements;
  $$ LANGUAGE sql IMMUTABLE;

CREATE OR REPLACE FUNCTION tsystem.function__drop_by_regex(
      _functionname_regex varchar,
      _namespace_regex varchar = '.*',
      _cascade bool = false,
      _commit bool = false
  ) RETURNS void AS $$
  DECLARE
      _function_data record;
      _query text;
  BEGIN

      -- _functionname_regex and _namespace_regex are put into a ^<exp>$ regex
      -- which results in exact matches. for unanchored matches you need to
      -- specify an expression like ".*abk.*" which drops every function with
      -- abk in its name.

      IF ( NOT _commit ) THEN
          RAISE NOTICE 'dry run:';
      END IF;

      FOR _function_data IN

        SELECT proname, nspname,
               pg_get_function_identity_arguments(p.oid) params
          FROM pg_proc p
          JOIN pg_namespace n ON p.pronamespace = n.oid
         WHERE proname ~* ('^' || _functionname_regex || '$')
           AND nspname ~* ('^' || _namespace_regex || '$')
      LOOP

          _query := format( '  DROP FUNCTION %I.%I(%s)', _function_data.nspname, _function_data.proname, _function_data.params );

          IF ( _cascade ) THEN
              _query := _query || ' CASCADE;';
          END IF;

          RAISE NOTICE '%', _query;

          IF ( _commit ) THEN
              EXECUTE _query;
          END IF;

      END LOOP;

  END $$ language plpgsql;

-- Gibt den Paramater des Connection Strings korrekt maskiert und umschlossen in einfachen Anführungsstrichen zurück.
-- Für die Maskierung werden C-Style Escapes (z.B. "'" -> "\'") benutzt.
-- Siehe https://www.postgresql.org/docs/current/libpq-connect.html#id-1.7.3.8.3.5.
-- Beispiel:
-- _connstr := concat(   'host=',      quote_literal__connstr_param( _host ),
--                       ' port=',     quote_literal__connstr_param( _port ),
--                       ' dbname=',   quote_literal__connstr_param( _dbname ),
--                       ' user=',     quote_literal__connstr_param( _user ),
--                       ' password=', quote_literal__connstr_param( _password ),
--                    );
CREATE OR REPLACE FUNCTION tsystem.quote_literal__connstr_param(
    IN _text varchar
  ) RETURNS varchar AS $$
  DECLARE
    _result varchar;
  BEGIN
    _result := REPLACE( REPLACE( _text, E'\\', E'\\\\' ), E'\'', E'\\\'');
    _result := concat(  '''',
                        _result,
                        ''''
                      );
    RETURN _result;
  END; $$ LANGUAGE plpgsql;


-- Ermittelt zu einem Eingangs-Datensatz (Tabelle + Primary Key) alle referenzierenden Einträge in anderen Tabellen
CREATE OR REPLACE FUNCTION tsystem.foreign_key__references__check(
    target_table      regclass,
    target_id         varchar
  )
  RETURNS TABLE(
    schema_name       text,
    table_name        text,
    column_name       text,
    on_update         text,
    on_delete         text,
    status            text,
    referencing_row   jsonb
  ) AS $$
  DECLARE
    r RECORD;
  BEGIN

    -- Alle Tabellen mit Referenz zur Eingangs-Tabelle durchlaufen
    FOR r IN
      WITH action_mapping (code, description) AS (
        VALUES
          ('a', 'no action'),
          ('r', 'restrict'),
          ('c', 'cascade'),
          ('n', 'set null'),
          ('d', 'set default')
      )
      SELECT nsp.nspname AS schema_name,
             cl.relname  AS table_name,
             a.attname   AS column_name,
             upd.description AS on_update,
             del.description AS on_delete
        FROM pg_constraint AS c
        JOIN pg_attribute  AS a   ON a.attnum = ANY(c.conkey) AND a.attrelid = c.conrelid
        JOIN pg_class      AS cl  ON cl.oid = c.conrelid
        JOIN pg_namespace  AS nsp ON nsp.oid = cl.relnamespace
        LEFT JOIN action_mapping upd ON confupdtype = upd.code
        LEFT JOIN action_mapping del ON confdeltype = del.code
       WHERE c.confrelid = target_table
         AND c.contype = 'f'
    LOOP

      -- Datensätze aus Tabellen mit Referenzen auf den Eingangs-Datensatz ausgeben
      RETURN QUERY EXECUTE format('
        SELECT %L AS schema_name,
               %L AS table_name,
               %L AS column_name,
               %L AS on_update,
               %L AS on_delete,
               ''Referenzen vorhanden'' AS status,
               row_to_json(t)::jsonb AS referencing_row
          FROM %I.%I t
         WHERE %I = $1', r.schema_name, r.table_name, r.column_name, r.on_update, r.on_delete, r.schema_name, r.table_name, r.column_name)
        USING target_id;

      -- Tabellen ohne Referenzen auf den Eingangs-Datensatz ausgeben
      RETURN QUERY EXECUTE format('
        SELECT %L AS schema_name,
               %L AS table_name,
               %L AS column_name,
               %L AS on_update,
               %L AS on_delete,
               ''Keine Referenzen'' AS status,
               NULL::jsonb AS referencing_row
         WHERE NOT EXISTS (
                SELECT 1
                  FROM %I.%I t
                 WHERE t.%I = $1
               )', r.schema_name, r.table_name, r.column_name, r.on_update, r.on_delete, r.schema_name, r.table_name, r.column_name, r.schema_name, r.table_name, r.column_name)
        USING target_id;

    END LOOP;

  END; $$ LANGUAGE plpgsql;

-- Castet einen Text in ein Datum um und gibt bei Fehlern den Default-Wert zurück
CREATE OR REPLACE FUNCTION TSystem.cast_date(
    text_date     varchar,
    default_value date DEFAULT null
  )
  RETURNS date AS $$
  BEGIN
    RETURN text_date::date;
  EXCEPTION WHEN OTHERS THEN
    RETURN default_value;
  END $$ LANGUAGE plpgsql IMMUTABLE PARALLEL SAFE;

--------------------------------------------------------------------------------
-- BEGIN DELEGATEDLOGIN
--
-- placed here, so everything within the delegatedlogin schema can be
-- purely restricted to master-login roles (directly, no inheritence)
-- thats why its also named delegatedlogin_xyz
-- following naming convention in that schema

--------------------------------------------------------------------------------
-- this needs to be run with SUPERUSER priviliges, which why SECURITY DEFINER
-- permissions are set in /1000-interfaces/99 Permissions.sql
CREATE OR REPLACE FUNCTION tsystem.delegatedlogin_masterlogin_revokerole(role_name text) RETURNS void AS $$
DECLARE
  _SQL varchar;

BEGIN
  -- must be session_user, the actually authenticated user that loged in to pg
  -- for security reasons
  _SQL = format('REVOKE %I FROM %I;', role_name, session_user);
  EXECUTE _SQL;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER;

--------------------------------------------------------------------------------
-- check rfid login, with provided user-role
-- used for very simple nfc-login (just the nfc chip and its uid) when loged in as a master-login role
-- or to allow authentication inside BDE-Module
CREATE OR REPLACE FUNCTION tsystem.delegatedlogin_check_rfid(role_name text, rfid text) RETURNS bool AS $$
DECLARE
  _user varchar;
  _ret  boolean;

BEGIN
  _ret = FALSE;

  if (NOT(role_name IS NULL) AND NOT(rfid IS NULL)) THEN
    SELECT
      ll_db_usename
    FROM
      llv
    WHERE
      ll_db_login
      AND COALESCE(ll_endd, current_date) <= current_date
      AND tsystem.enum_getvalue(ll_rfid, rfid::character varying, ','::character varying) -- list to check against, value to check, separator
    INTO
      _user
    ;

    _ret = (_user = role_name);
  END IF;

  RETURN _ret;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER;
-- END DELEGATEDLOGIN
--------------------------------------------------------------------------------

--- #22403 Start
-- Eintrag im Windows-Schlüsselbund hinzufügen
SELECT tsystem.function__drop_by_regex( 'credential__set', 'TSystem_security', _commit => true );
CREATE OR REPLACE FUNCTION TSystem_security.credential__set(
      servicename varchar,
      username varchar,
      userpassword varchar,
      overwrite_pw boolean DEFAULT false
  )
RETURNS boolean
AS $$
  import keyring as kr
  if kr.get_password(servicename, username) != None:
      if overwrite_pw == True:
          kr.delete_password(servicename, username)
          kr.set_password(servicename, username, userpassword)
          return True
      else:
          return False
  else:
      kr.set_password(servicename, username, userpassword)
      return True
$$ LANGUAGE plpython3u;

REVOKE EXECUTE ON FUNCTION TSystem_security.credential__set(character varying, character varying, character varying, boolean) FROM PUBLIC;

-- SELECT TSystem_security.credential__set( servicename => 'SyncroKonto', username => 'syncro', userpassword => 'geheim', overwrite_pw => true );
-- SELECT TSystem_security.credential__set( servicename => 'PostgresKonto', username => 'postgres', userpassword => 'super geheim', overwrite_pw => true );

-- Passwort aus entsprechenden Eintrag im Windows-Schlüsselbund auslesen
SELECT tsystem.function__drop_by_regex( 'credential__get', 'TSystem_security', _commit => true );
CREATE OR REPLACE FUNCTION TSystem_security.credential__get(
      servicename varchar,
      username varchar
  )
RETURNS varchar
AS $$
  import keyring as kr
  if kr.get_password(servicename, username) != None:
      ps = str(kr.get_password(servicename, username))
  else:
      return

  return ps
$$ LANGUAGE plpython3u;

REVOKE EXECUTE ON FUNCTION TSystem_security.credential__get(character varying, character varying) FROM PUBLIC;

-- SELECT TSystem_security.credential__get( servicename => 'SyncroKonto', username => 'syncro' );
-- SELECT TSystem_security.credential__get( servicename => 'PostgresKonto', username => 'postgres' );

-- Eintrag im Windows-Schlüsselbund löschen
SELECT tsystem.function__drop_by_regex( 'credential__delete', 'TSystem_security', _commit => true );
CREATE OR REPLACE FUNCTION TSystem_security.credential__delete(
      servicename varchar,
      username varchar
  )
RETURNS boolean
AS $$
  import keyring as kr
  if kr.get_password(servicename, username) != None:
      kr.delete_password(servicename, username)
      return True
  else:
      return False
$$ LANGUAGE plpython3u;

REVOKE EXECUTE ON FUNCTION TSystem_security.credential__delete(character varying, character varying) FROM PUBLIC;

-- SELECT TSystem_security.credential__delete( servicename => 'SyncroKonto', username => 'syncro' );
-- SELECT TSystem_security.credential__delete( servicename => 'PostgresKonto', username => 'postgres' );

-- alle vorhandenen Einträge im Windows-Schlüsselbund auflisten
SELECT tsystem.function__drop_by_regex( 'credentials__list', 'TSystem_security', _commit => true );
CREATE OR REPLACE FUNCTION TSystem_security.credentials__list( out servicename varchar, out username varchar )
RETURNS setof record
AS $$
  from win32 import win32cred
  try:
    credentials = win32cred.CredEnumerate(None, 0)
  except Exception as e:
    if e.winerror == 1168:
      plpy.notice('No credentials stored.')
      return None
    else:
      raise
  for credential in credentials:
    TargetName = credential["TargetName"]
    UserName = credential["UserName"]
    yield ( [TargetName, UserName] )
$$ LANGUAGE plpython3u;

REVOKE EXECUTE ON FUNCTION TSystem_security.credentials__list() FROM PUBLIC;

-- SELECT * FROM TSystem_security.credentials__list() ORDER BY servicename, username;
--- #22403 Ende